<?php

namespace Acelle\Jobs;

use Acelle\Model\AutoTrigger;
use Acelle\Model\Subscriber;
use Acelle\Model\SendingServer;
use Acelle\Model\Subscription;
use Acelle\Library\Exception\RateLimitExceeded;
use Acelle\Library\RouletteWheel;
use Throwable;

class SendSingleMessage extends SendMessage
{
    protected $autoTriggerId;
    protected $autoTrigger;
    protected $actionId;
    protected $action;

    protected $deliveryLogger;

    public function __construct($campaign, Subscriber $subscriber, RouletteWheel $servers, Subscription $subscription = null, $autoTriggerId = null, $actionId = null)
    {
        parent::__construct($campaign, $subscriber, $servers, $subscription);

        $this->autoTriggerId = $autoTriggerId;
        $this->actionId = $actionId;
    }

    public function afterSuccess()
    {
        $this->deliveryLogger()->info('Updating trigger status (sent_at)');

        $this->getAutoTrigger()->withUpdateLock(function ($trigger) {
            // Mark the related trigger action as done

            $action = $this->getAction();
            $action->setSent();
            $this->deliveryLogger()->info('Updated trigger status (sent_at)');
        });

        $this->deliveryLogger()->info('Run check() again');

        // The check() method is already wrapped in a cache lock
        // no need to pass $manually = true here because
        // this action is only to confirm the completion of Send (execute returns true, then last_executed_time is recorded, done)
        $this->getAutoTrigger()->check();
    }

    public function getAutoTrigger()
    {
        if (is_null($this->autoTrigger)) {
            if ($this->customer->hasLocalDb()) {
                $connection = $this->customer->db_connection;
            } else {
                $connection = 'mysql';
            }

            $this->autoTrigger = AutoTrigger::on($connection)->find($this->autoTriggerId);
        }

        if (is_null($this->autoTrigger)) {
            throw new \Exception("Cannot find trigger {$this->autoTriggerId} (DB connection ".\DB::connection()->getName().")");
        }

        return $this->autoTrigger;
    }

    public function getAction()
    {
        if (is_null($this->action)) {
            $this->action = $this->getAutoTrigger()->getActionById($this->actionId);
        }

        return $this->action;
    }

    public function handle()
    {
        $this->startAt = now()->getTimestampMs();
        $this->customer->setUserDbConnection();
        $this->send();
    }

    public function handleRateLimitExceeded($email, RateLimitExceeded $ex)
    {
        $secondsToDelay = 600;
        $now = now()->getTimestampMs();
        $time = ($now - $this->startAt) / 1000;
        $this->deliveryLogger()->warning(sprintf("Delay [%s] for %s seconds (no batch): %s (it taks %s to get here!)", $email, $secondsToDelay, $ex->getMessage(), $time));

        $this->getAutoTrigger()->withUpdateLock(function ($trigger) use ($email, $ex, $secondsToDelay) {
            $delayNote = sprintf("Delayed for %s seconds: %s", $secondsToDelay, $ex->getMessage());

            $action = $this->getAction();
            $action->setOption('delay_note', $delayNote);
            $action->save();
        });

        $this->release($secondsToDelay); // should be only 60 seconds
        return $secondsToDelay;
    }

    public function handleUnknownException($ex)
    {
        $this->getAutoTrigger()->withUpdateLock(function ($trigger) use ($ex) {
            // Mark the related trigger action as done
            $action = $this->getAction();
            $action->setError($ex->getMessage());
            $this->deliveryLogger()->info('Error sending trigger email: '.$ex->getMessage());
        });

        // Return true is important, telling the caller not to throw exception
        return true;
    }

    public function deliveryLogger()
    {
        if (is_null($this->deliveryLogger)) {
            $this->deliveryLogger = applog("send-message-{$this->campaign->uid}");
        }
        return $this->deliveryLogger;
    }
}
